/************************************************************************************************************\

Module Name:    LVideoIn.h

Description:    The LVideoIn module of the API provides the programming interface for the video input
                hardware.

References:     LVideoIn.doc Revision 0.21.

    Copyright (c) 2016, Matrox Graphics Inc.
    All Rights Reserved.

\************************************************************************************************************/

#ifndef INC_LVIDEOIN_H
#define INC_LVIDEOIN_H

// -----------------------------------------------------------------------------------------------------------
//                                   I N C L U D E S   A N D   U S I N G S
// -----------------------------------------------------------------------------------------------------------

#include "LVideo.h"

#ifdef EXTERNAL_HDCP_SUPPORT
#include "LSecureEngine.h"
#endif //EXTERNAL_HDCP_SUPPORT

typedef struct LVIDEOIN_OBJECT*      LVideoIn_Handle;

// -----------------------------------------------------------------------------------------------------------
//                                                U S A G E
// -----------------------------------------------------------------------------------------------------------

/*  Capture buffers on a Video input.

    #include "Liberatus.h"
    #include "LBuffer.h"
    #include "LVideoIn.h"

    int main()
    {
        LDevice_Handle              hDevice         = MNULL;
        LBuffer_Handle              hCapturedBuffer = MNULL;
        LVideoIn_Handle             hVideoIn        = MNULL;
        MBOOL32                     bSourceValid    = MFALSE;
        MUINT32                     uiWidth         = 0;
        MUINT32                     uiHeight        = 0;
        LPixelFormat                ePixelFormat    = LPixelFormat_INVALID;
        MUINT64                     uiFieldCounter  = 0;
        MUINT64                     uiTickRefCounter= 0;
        MBOOL32                     bIsTopField     = 0;
        MUINT32                     uiIndex         = 0;

        Liberatus_Load();
        
        hDevice = Liberatus_GetDevice(0);           // Primary device handle

        LVideoIn_GetHandle(hDevice, 
                           0,                       // Handle on the first usable resource
                           LAccessMode_READWRITE_EXCLUSIVE,
                           &hVideoIn);

        LVideoIn_DetectSource(hVideoIn,
                              &bSourceValid,
                              MNULL,
                              &uiWidth,
                              &uiHeight);

        if (bSourceValid)
        {

            LVideoIn_EnumSupportedPixelFormat(hVideoIn, 
                                              0,
                                              &ePixelFormat);

            LVideoIn_CreateBuffers(hVideoIn, 
                                   uiWidth,
                                   uiHeight,
                                   ePixelFormat,
                                   100);                   // 100 Buffers max.

            LVideoIn_StartCapture(hVideoIn);
            
            for (uiIndex=0; uiIndex < 100; uiIndex++)
            {
            
                LVideoIn_GetNextBuffer( hVideoIn,
                                        MTRUE,                      // Wait for Ready buffer.
                                        200,                        // in ms
                                        &hCapturedBuffer,
                                        &uiFieldCounter,
                                        &uiTickRefCounter,
                                        &bIsTopField);

                // USE VIDEO BUFFER HERE

                LVideoIn_ReleaseBuffer( hVideoIn,
                                        hCapturedBuffer);

            }   

            LVideoIn_StopCapture(hVideoIn);

            LVideoIn_DestroyBuffers(hVideoIn);

        }

        LVideoIn_ReleaseHandle(hVideoIn);

        Liberatus_UnLoad();
        
        return 0;
    }
*/
// -----------------------------------------------------------------------------------------------------------
//                                               O V E R V I E W
// -----------------------------------------------------------------------------------------------------------
/************************************************************************************************************\

    A video input can be used to capture video input stream data (video, audio, and/or ancillary) into
    LBuffer objects (referred to as capture buffers from here on) that can then be used to perform processing 
    on the captured data.

    A video input implements an internal state machine (see Figure 1), whose state may be queried by
    LVideoIn_GetCaptureState().

    Here the video input state machine diagram.


                                      +----------------------------------+
                                      |                                  |
                                      |                                  |
                                      |  LVideoIn_State_NOT_INITIALIZED  |
                                      |                                  |
                                      |                                  |
                                      +----------------------------------+
                                         |                         /\
                                         |                         |
             LVideoIn_DetectSource() +   |                         |
             LVideoIn_CreateBuffers()+   |                         | LVideoIn_DestroyBuffers()
             LVideoIn_StartCapture()     |                         |
                         ________________/                          \__________
                        /                                                      \
                        |                                                      |
                        |                                                      |
                        \/                                                     |
    +----------------------------+     LVideoIn_StopCapture() or    +------------------------------+
    |                            |   LVideoIn_Event_SOURCE_CHANGED  |                              |
    |                            |--------------------------------->|                              |
    |   LVideoIn_State_STARTED   |                                  |    LVideoIn_State_STOPPED    |
    |                            |  LVideoIn_StartCapture()         |                              |
    |                            |<---------------------------------|                              |
    +----------------------------+                                  +------------------------------+

    Figure 1: LVideo capture state


    A video input has an internal set of capture buffers that are used to capture the input stream data
    and that are loaned to a Liberatus application for processing.

    Each video input internal capture buffer has two (2) different attributes that describe its current
    state.

    The first attribute is called Ready and represents whether or not it contains all captured stream
    data. The second attribute is called Loaned and represents whether or not a capture buffer is loaned
    to a Liberatus application for processing.

    Figure 2 shows when a capture buffer Ready and Loaned state attributes change.


    Ready state attribute:
    ======================
        +----------------------+                                     +----------------------+
        |    Capture Buffer    |     Stream Captured Data            |    Capture Buffer    |
        |                      |------------------------------------>|                      |
        |        EMPTY         |                                     |        READY         |
        |                      |<------------------------------------|                      |
        |                      |        ReleaseBuffer() or           |                      |
        +----------------------+      DiscardReadyBuffers() or       +----------------------+
                                     ScheduleReleaseBuffer()

    Loaned state attribute:
    =======================
        +----------------------+                                     +----------------------+
        |    Capture Buffer    |           GetNextBuffer()           |    Capture Buffer    |
        |                      |------------------------------------>|                      |
        |      RELEASED        |                                     |        LOANED        |
        |                      |<------------------------------------|                      |
        |                      |           ReleaseBuffer() or        |                      |
        +----------------------+        ScheduleReleaseBuffer()      +----------------------+

    Figure 2: LVideo capture buffer state


    Attributes states meaning:
    ==========================
    
    +-------------+-------------+-----------------------------------------------------------+
    |   Ready     |   Loaned    | Meaning                                                   |
    +-------------+-------------+-----------------------------------------------------------+
    |   MFALSE    |   MFALSE    | Capture buffer contains no stream data and isn't loaned to|
    |             |             | a Liberatus application. The video input can capture input|
    |             |             | stream data in the buffer.                                |
    +-------------+-------------+-----------------------------------------------------------+
    |   MFALSE    |   MTRUE     | Capture buffer contains no stream data and is loaned to a |
    |             |             | Liberatus application. The video input can capture input  |
    |             |             | stream data in the buffer, and the Liberatus application  |
    |             |             | can schedule operations on the capture buffer.            |
    +-------------+-------------+-----------------------------------------------------------+
    |   MTRUE     |   MFALSE    | Capture buffer contains stream data and isn't loaned to a |
    |             |             | Liberatus application. The video input can't capture input|
    |             |             | stream data in the buffer.                                |
    +-------------+-------------+-----------------------------------------------------------+
    |   MTRUE     |   MTRUE     | Capture buffer contains stream data and is loaned to a    |
    |             |             | Liberatus application. The video input can't capture input|
    |             |             | stream data in the buffer and the Liberatus application   |
    |             |             | can schedule or perform direct operations on the capture  |
    |             |             | buffer.                                                   |
    +-------------+-------------+-----------------------------------------------------------+

        Capture Buffer pool definitions

            <Total>        Count of all buffers in the pool.

            <Ready>        Count of buffers containing captured data.
            <Empty>        Count of buffers containing no captured data. <Total - Ready>

            <Loaned>       Count of buffers loaned to the application.
            <Released>     Count of buffers not loaned to the application. <Total - Loaned>

\************************************************************************************************************/


// -----------------------------------------------------------------------------------------------------------
//                                   C O N S T A N T S   A N D   T Y P E S
// -----------------------------------------------------------------------------------------------------------  

/************************************************************************************************************\

Enum:           LVideoIn_ControlId

Description:    Enumerates all possible video input device control IDs.

Comments:       - For more information about controls, see the LVideoIn_ControlInfo structure.
                - Some defined controls may be not be supported by the current hardware configuration.  To 
                  determine whether a given control is supported or not by the current hardware configuration, 
                  use the LVideoIn_GetControlInfo service.
                - The sub-field mode use for low latency, if supported, has to be set before starting the
                  capture.

\************************************************************************************************************/
typedef enum
{
    LVideoIn_ControlId_INVALID                  = 0x00000000,   // Invalid value.

    LVideoIn_ControlId_SUB_FIELD_MODE           = 0x00000001,   // Enable sub field mode.
                                                                //  0 Disable sub-field mode.
                                                                //  1 Enable half field mode.
                                                                //  2 Enable quarter field mode. 
                                                                //  3 Enable eighth field mode.
                                                                //  4 Enable sixteenth field mode.

    LVideoIn_ControlId_ANALOG_ADJUST_PHASE      = 0x00000010,   // Analog pixels phase adjustment.
    LVideoIn_ControlId_ANALOG_ADJUST_ALIGN      = 0x00000011,   // Analog pixels alignment.

    LVideoIn_ControlId_FRAME_RATE_DECIMATION    = 0x00000020,   // Input source frame rate decimation control. 
                                                                //  The decoder reduces the frame rate of 
                                                                //  the input source. The value is specified 
                                                                //  in 'N' fields (or frames) per second.
    
    LVideoIn_ControlId_PRESCALING_H             = 0x00000030,   // Horizontal input source shrinking.
    LVideoIn_ControlId_PRESCALING_V             = 0x00000031,   // Vertical input source shrinking.

    LVideoIn_ControlId_POSITION_H               = 0x00000040,   // Horizontal capture position in pixels 
                                                                //  control.
    LVideoIn_ControlId_POSITION_V               = 0x00000041,   // Vertical capture position in lines control.

    LVideoIn_ControlId_BRIGHTNESS               = 0x00000100,   // Brightness control.
    LVideoIn_ControlId_BRIGHTNESS_RED           = 0x00000101,   // Brightness control for red component.
    LVideoIn_ControlId_BRIGHTNESS_GREEN         = 0x00000102,   // Brightness control for green component.
    LVideoIn_ControlId_BRIGHTNESS_BLUE          = 0x00000103,   // Brightness control for blue component.

    LVideoIn_ControlId_CONTRAST                 = 0x00000110,   // Contrast control.
    LVideoIn_ControlId_CONTRAST_RED             = 0x00000111,   // Contrast control for red component.
    LVideoIn_ControlId_CONTRAST_GREEN           = 0x00000112,   // Contrast control for green component.
    LVideoIn_ControlId_CONTRAST_BLUE            = 0x00000113,   // Contrast control for blue component.

    LVideoIn_ControlId_SATURATION               = 0x00000120,   // Saturation control.

    LVideoIn_ControlId_HUE                      = 0x00000130,   // Hue control.

    LVideoIn_ControlId_CSC                      = 0x00000200,   // Color space conversion.
    LVideoIn_ControlId_CSC_PARAMETER1           = 0x00000201,   // Color space conversion parameters 1.
    LVideoIn_ControlId_CSC_PARAMETER2           = 0x00000202,   // Color space conversion parameters 2.
    LVideoIn_ControlId_CSC_PARAMETER3           = 0x00000203,   // Color space conversion parameters 3.
    LVideoIn_ControlId_CSC_PARAMETER4           = 0x00000204,   // Color space conversion parameters 4.

    LVideoIn_ControlId_FILTER                   = 0x00000800,   // General filter control. TBD

    LVideoIn_ControlId_FORCE32                  = 0x7FFFFFFF,   // Dummy value to force to use 32-bits. 

} LVideoIn_ControlId;

/************************************************************************************************************\

Enum:           LVideoIn_State

Description:    Enumerates all possible capture state values.

Comments:       None.

\************************************************************************************************************/
typedef enum
{
    LVideoIn_State_INVALID                  = 0x00000000,   // Invalid value.
    LVideoIn_State_NOT_INITIALIZED          = 0x00000001,   // Capture isn't initialized and can't be
                                                            //  started until it's properly initialized.
    LVideoIn_State_STARTED                  = 0x00000002,   // Capture already started, fields are being 
                                                            //  captured.
    LVideoIn_State_STOPPED                  = 0x00000004,   // Capture has stopped, but it is still 
                                                            //  initialized and can be restarted.
    LVideoIn_State_FORCE32                  = 0x7FFFFFFF    // Dummy value to force to use 32-bits. 
} LVideoIn_State;


/************************************************************************************************************\

Enum:           LVideoIn_Event

Description:    Enumerates all possible capture events.

Comments:       LHandleVideoIn.h defines valid mask as 0xFFFF. So only 16-lsb are valid for events.

\************************************************************************************************************/
typedef enum
{
    LVideoIn_Event_INVALID              = 0x00000000,   // Invalid value.

    LVideoIn_Event_STARTED              = 0x00000001,   // Capture already started, fields are being 
                                                        //  captured.

    LVideoIn_Event_STOPPED              = 0x00000002,   // Capture has stopped, but it's still 
                                                        //  initialized and can be restarted.

    LVideoIn_Event_SOURCE_CHANGED       = 0x00000004,   // Input source has changed.  The capture was stopped 
                                                        //  and must be re-initialized to capture fields again.

    LVideoIn_Event_OUT_OF_BUFFER        = 0x00000008,   // Capture is forced to drop fields because no empty
                                                        //  capture buffer is available.  Capture  
                                                        //  automatically resumes when capture buffers become
                                                        //  available again.
    
    LVideoIn_Event_ALL_BUFFER_RELEASED  = 0x00000010,   // Capture is stopped and all allocated capture
                                                        //  buffers are released and emptied. This event can
                                                        //  be used to figure out when the buffers may be
                                                        //  destroyed safely using LVideoIn_DestroyBuffers()
                                                        //  without causing a resource leaks.

    // LVideoIn_Event_UNUSED            = 0x00000020,   // This event is unused for now.

    // LVideoIn_Event_UNUSED            = 0x00000040,   // This event is unused for now.

    LVideoIn_Event_SOURCE_DETECTED      = 0x00000080,   // Input source has been detected.
                                                        //  LVideoIn_GetDetectedVideoParameters should return
                                                        //  new parameters.

    LVideoIn_Event_BLANKED              = 0x00000100,   // Input has been blanked (protection).

    LVideoIn_Event_UNBLANKED            = 0x00000200,   // Input has been unblanked (protection).

    LVideoIn_Event_FORCE32              = 0x7FFFFFFF    // Dummy value to force to use 32-bits.
} LVideoIn_Event;

/************************************************************************************************************\

Struct:         LVideoIn_ControlInfo

Description:    Structure defining a video input control information.

Comments:       - A control is an attribute of the video input device with a default value, a minimal value, 
                  maximal value, and a step value.
                - The minimal value and the step value should be used to compute all valid values within the 
                  range of supported control values.
                - For more information on available controls, see LVideoIn_ControlId.
                - To manipulate the controls value, see the LVideoIn_ControlInfo, LVideoIn_GetControlValues 
                  or LVideoIn_SetControlValues.
                - To get the current value of a control use the LVideoIn_GetControlValues service.


\************************************************************************************************************/
typedef struct tagLVideoIn_ControlInfo
{
    MINT32  iMinimum;   // Minimum value.
    MINT32  iMaximum;   // Maximum value.
    MINT32  iStep;      // Step value.
    MINT32  iDefault;   // Default value.
} LVideoIn_ControlInfo;

/************************************************************************************************************\

Struct:         LVideoIn_InfoFrameHeader

Description:    Structure defining an AVI InfoFrame header.

Comments:       None.

\************************************************************************************************************/
typedef struct tagLVideoIn_InfoFrameHeader
{
    MUINT32 uiType;     // AVI InfoFrame packet ID value.
    MUINT32 uiVersion;  // AVI InfoFrame packet version value.
    MUINT32 uiLength;   // AVI InfoFrame packet length value.
} LVideoIn_InfoFrameHeader;

// -----------------------------------------------------------------------------------------------------------
//                         G L O B A L   F U N C T I O N   D E C L A R A T I O N S
// -----------------------------------------------------------------------------------------------------------


#if defined (__cplusplus)
extern "C" {
#endif

/************************************************************************************************************\

Function:       LVideoIn_GetCount

Description:    Gets the number of independent video inputs available for the specified device.

Parameters:     IN  hDev            Device handle.
                OUT puiCount        Pointer to an MUINT32 variable to be filled with the number of independent 
                                    video inputs available.

Return Value:   LStatus_OK              Function completed successfully.
                LStatus_INVALID_PARAM   Function failed due to invalid parameters.
                LStatus_FAIL            Function failed.

Comments:       None.

\************************************************************************************************************/
LAPI LStatus LVideoIn_GetCount(
                LDevice_Handle      hDev, 
                MUINT32*            puiCount); 


/************************************************************************************************************\

Function:       LVideoIn_GetHandle

Description:    Gets the video input handle for the specified independent video input device.

Parameters:     IN  hDev            Device handle.
                IN  uiIndex         Index of the independent video input for which to return the handle.
                IN  eAccessMode     Requested access mode.  Must be a value from LAccessMode.
                OUT phVideoIn       Pointer to a LVideoIn_Handle object to be filled with video input
                                    device handle on success. MNULL otherwise.


Return Value:   LStatus_OK                  Function completed successfully.
                LStatus_INVALID_PARAM       Function failed due to invalid parameters.
                LStatus_FAIL                Function failed.
                LStatus_UNSUPPORTED         Invalid video input device index.  No hardware associated.
                LStatus_OUT_OF_RESOURCES    Specified video input device can't be used by the Liberatus
                                            applications because it's already being used by the system.
                LStatus_RESOURCES_BUSY      Video input handle couldn't be returned because the video input
                                            is busy.

Comments:       - If eAccessMode is LAccessMode_READONLY, functions that modify the video input can't be 
                  called.
                  If eAccessMode is LAccessMode_READWRITE or LAccessMode_READWRITE_EXCLUSIVE, functions that 
                  modify the video input can be called.
                - Only one handle with LAccessMode_READWRITE_EXCLUSIVE read/write access mode can exist at 
                  any one time. If a handle with read/write access has already been granted to a caller, 
                  any attempt to get another handle with read/write access returns a failure.
                - If eAccessMode is LAccessMode_READWRITE_EXCLUSIVE the video capture will be automatically
                  stopped and all buffers will be released when the handle is released.

\************************************************************************************************************/
LAPI LStatus LVideoIn_GetHandle(
                LDevice_Handle          hDev, 
                MUINT32                 uiIndex, 
                LAccessMode             eAccessMode,
                LVideoIn_Handle*        phVideoIn);


/************************************************************************************************************\

Function:       LVideoIn_ReleaseHandle

Description:    Releases a handle to an independent video input device.

Parameters:     IN  hVideoIn            Handle to the video input device.

Return Value:   LStatus_OK              Function completed successfully.
                LStatus_INVALID_PARAM   Function failed due to invalid parameters.
                LStatus_FAIL            Function failed.

Comments:       - If eAccessMode is LAccessMode_READWRITE_EXCLUSIVE, the video capture will be automatically
                  stopped and all buffers will be released when the handle is released.

\************************************************************************************************************/
LAPI LStatus LVideoIn_ReleaseHandle(
                LVideoIn_Handle     hVideoIn);


/************************************************************************************************************\

Function:       LVideoIn_EnumSignalType

Description:    Enumerates all supported video input signal types. 

Parameters:     IN  hVideoIn            Handle to the video input device.
                IN  uiSignalIndex       Index of the signal type to enumerate.
                OUT peSignalType        Pointer to an LVideo_SignalType object where the enumerated signal
                                        types are written.

Return Value:   LStatus_OK              Function completed successfully.
                LStatus_INVALID_PARAM   Function failed due to invalid parameters.
                LStatus_FAIL            Function failed.
                LStatus_NO_MORE_DATA    No signal type value is available for the specified index.
                
Comments:       None.

\************************************************************************************************************/
LAPI LStatus LVideoIn_EnumSignalType(
                LVideoIn_Handle         hVideoIn, 
                MUINT32                 uiSignalIndex,
                LVideo_SignalType*      peSignalType);


/************************************************************************************************************\

Function:       LVideoIn_GetDefaultSignalType

Description:    Gets the default signal type of the video input device.

Parameters:     IN  hVideoIn            Handle to the video input device.
                OUT peSignalType        Pointer to an LVideo_SignalType object where the enumerated signal
                                        types are written.

Return Value:   LStatus_OK              Function completed successfully.
                LStatus_INVALID_PARAM   Function failed due to invalid parameters.
                LStatus_FAIL            Function failed.

Comments:       - The default video input signal type device may change after an input source detection.

\************************************************************************************************************/
LAPI LStatus LVideoIn_GetDefaultSignalType(
                LVideoIn_Handle         hVideoIn, 
                LVideo_SignalType*      peSignalType);


/************************************************************************************************************\

Function:       LVideoIn_SetSignalType

Description:    Sets the video input device signal type of the video input device.

Parameters:     IN  hVideoIn            Handle to the video input device.
                IN  eSignalType         Video signal type to set for the independent video input.  Must be a 
                                        value from the LVideo_SignalType.

Return Value:   LStatus_OK              Function completed successfully.
                LStatus_INVALID_PARAM   Function failed due to invalid parameters.
                LStatus_FAIL            Function failed.
                LStatus_ACCESS_DENIED   The hVideoIn handle is READONLY.  Read/write access
                                        (LAccessMode_READWRITE) is needed to call this function.

Comments:       - Setting the signal type may reserve and program some hardware resources that are shared 
                  with other video input device.

\************************************************************************************************************/
LAPI LStatus LVideoIn_SetSignalType(
                LVideoIn_Handle     hVideoIn, 
                LVideo_SignalType   eSignalType);


/************************************************************************************************************\

Function:       LVideoIn_GetSignalType

Description:    Gets the current video input device signal type.

Parameters:     IN  hVideoIn            Handle to the video input device.
                OUT peSignalType        Pointer to an LVideo_SignalType object where the enumerated signal
                                        types are written.

Return Value:   LStatus_OK              Function completed successfully.
                LStatus_INVALID_PARAM   Function failed due to invalid parameters.
                LStatus_FAIL            Function failed.

Comments:       None.

\************************************************************************************************************/
LAPI LStatus LVideoIn_GetSignalType(
                LVideoIn_Handle     hVideoIn, 
                LVideo_SignalType*  peSignalType);


/************************************************************************************************************\

Function:       LVideoIn_DetectSource

Description:    Performs an input source mode detection based on the current signal type. 

Parameters:     IN  hVideoIn            Handle to the video input device.
                OUT pbIsCapturable      Pointer to an MBOOL32 variable to be filled with MTRUE if the detected 
                                        source can be captured and MFALSE if no source is detected or the 
                                        detected source is outside the range of sources that can be captured.
                OUT pbIsDetected        Pointer to an MBOOL32 variable to be filled with MTRUE if a source, 
                                        whether or not it can be captured, is detected, and MFALSE otherwise. 
                                        Can be MNULL if source detected state value isn't requested.
                OUT puiWidth            Pointer to an MUINT32 variable to be filled with the width, in pixels, 
                                        of the detected source. Can be MNULL if the width value isn't 
                                        requested.
                OUT puiHeight           Pointer to an MUINT32 variable to be filled with the height, in lines,
                                        of the detected source. Can be MNULL if the height value isn't 
                                        requested.

Return Value:   LStatus_OK              Function completed successfully.
                LStatus_INVALID_PARAM   Function failed due to invalid parameters.
                LStatus_FAIL            Function failed.
                LStatus_BUSY            Detection impossible because the capture is running.
                LStatus_ACCESS_DENIED   The hVideoIn handle is READONLY.  Read/write access
                                        (LAccessMode_READWRITE) is needed to call this function.

Comments:       - Input source detection is impossible while the capture is running. The service then returns 
                  an LStatus_BUSY error code.
                - Returned detected source width and height values can be used to set proper buffer attributes 
                  via the LVideoIn_CreateBuffers service.
                - In interlaced mode, the height returned represents the number of lines in the field
                  (for example, 540 lines in 1080i).
                - To retrieve more information on the detected input source, use the 
                  LVideoIn_GetDetectedVideoParameters service.

\************************************************************************************************************/
LAPI LStatus LVideoIn_DetectSource(
                LVideoIn_Handle     hVideoIn,
                MBOOL32*            pbIsCapturable,
                MBOOL32*            pbIsDetected,
                MUINT32*            puiWidth,
                MUINT32*            puiHeight);


/************************************************************************************************************\

Function:       LVideoIn_GetDetectedVideoParameters

Description:    Gets the detected input source mode video parameters.

Parameters:     IN  hVideoIn            Handle to the video input device.
                OUT poVidParam          Pointer to a LVideo_VidParam variable to be filled with detected video
                                        parameters.

Return Value:   LStatus_OK              Function completed successfully.
                LStatus_INVALID_PARAM   Function failed due to invalid parameters.
                LStatus_FAIL            Function failed.
                LStatus_NOT_OPTIMAL     Function doesn't return the latest video parameters.

Comments:       - Can be called with a (LAccessMode_READONLY) LVideoIn_Handle.
                - LStatus_NOT_OPTIMAL: the function returns the cached video parameters,
                  LVideoIn_DetectSource must be called to detect new video parameters.
                - LVideoIn_Event_SOURCE_DETECTED event indicates that new video parameters can be retrieved.
                - poVidParam->uiHActive set to 0 when no source is detected.
                - No guarantee that the detected source can be captured. LVideoIn_DetectSource returns this
                  information.

\************************************************************************************************************/
LAPI LStatus LVideoIn_GetDetectedVideoParameters(
                LVideoIn_Handle     hVideoIn, 
                LVideo_VidParam*    poVidParam);


/************************************************************************************************************\

Function:       LVideoIn_EnumSupportedPixelFormat

Description:    Enumerates the pixel formats supported for capture from the current input source.

Parameters:     IN  hVideoIn            Handle to the video input device.
                IN  uiIndex             Index of the pixel format to be enumerated.
                OUT pePixelFormat       Pointer to an LPixelFormat object to be filled with the enumerated
                                        pixel format.

Return Value:   LStatus_OK              Function completed successfully.
                LStatus_INVALID_PARAM   Function failed due to invalid parameters.
                LStatus_FAIL            Function failed.
                LStatus_NO_MORE_DATA    No pixel format value is available for the specified index.

Comments:       None.

\************************************************************************************************************/
LAPI LStatus LVideoIn_EnumSupportedPixelFormat(
                LVideoIn_Handle     hVideoIn, 
                MUINT32             uiIndex,
                LPixelFormat*       pePixelFormat);


/************************************************************************************************************\

Function:       LVideoIn_CreateBuffers

Description:    Creates video capture buffers.

Parameters:     IN  hVideoIn            Handle to the video input device.
                IN  uiWidth             Maximum width, in pixels, of the video buffer to capture.
                IN  uiHeight            Maximum height, in lines, of the video buffer to capture.
                IN  ePixelFormat        Pixel format of the video buffer to capture. Must be a value from
                                        LPixelFormat.
                IN  uiNbBuffers         Number of buffers allocated for the capture.

Return Value:   LStatus_OK              Function completed successfully.
                LStatus_INVALID_PARAM   Function failed due to invalid parameters.
                LStatus_FAIL            Function failed.
                LStatus_ACCESS_DENIED   The hVideoIn handle is READONLY.  Read/write access
                                        (LAccessMode_READWRITE) is needed to call this function.

Comments:       - Buffer attributes can't be changed while the capture is running.
                - The video input device allocates the specified number of buffers before the capture 
                  is started.  
                - Recommended minimum number of buffers is two (2) if the time taken to process a captured
                  buffer is less than the time taken to capture a buffer. Otherwise, a larger minimum number
                  of buffers should be used.
                - Recommended number of buffers depends on the time the application needs to process
                  the captured buffers.  There's no fixed maximum, only the available memory limits the 
                  maximum.
                - Extra buffers will be created for hardware processing. The number of extra buffers created
                  may be queried using LVideoIn_GetCaptureState().

\************************************************************************************************************/
LAPI LStatus LVideoIn_CreateBuffers(
                LVideoIn_Handle hVideoIn, 
                MUINT32         uiWidth,
                MUINT32         uiHeight,
                LPixelFormat    ePixelFormat,
                MUINT32         uiNbBuffers);

/************************************************************************************************************\

Function:       LVideoIn_DestroyBuffers

Description:    Destroys video capture buffers.

Parameters:     IN  hVideoIn            Handle to the video input device.

Return Value:   LStatus_OK              Function completed successfully.
                LStatus_INVALID_PARAM   Function failed due to invalid parameters.
                LStatus_FAIL            Function failed.
                LStatus_ACCESS_DENIED   The hVideoIn handle is READONLY.  Read/write access
                                        (LAccessMode_READWRITE) is needed to call this function.

Comments:       - It's important to stop the capture and release the buffer before destroying the buffer.
                - You need to destroy a buffer before changing its attributes.
                - The buffers can be released out of order over GetNextBuffer.


\************************************************************************************************************/
LAPI LStatus LVideoIn_DestroyBuffers(
                LVideoIn_Handle     hVideoIn);


/************************************************************************************************************\

Function:       LVideoIn_GetBufferAttributes

Description:    Gets the current capture video buffer attributes. 

Parameters:     IN  hVideoIn            Handle to the video input device.
                OUT uiWidth             Pointer to an MUINT32 variable to be filled with the width, in pixels,
                                        of the capture video buffer.
                OUT uiHeight            Pointer to an MUINT32 variable to be filled with the height, in lines,
                                        of the capture video buffer.
                OUT ePixelFormat        Pointer to an LPixelFormat object to be filled with the pixel format
                                        of the capture video buffer.
                OUT puiNbBuffers        Pointer to an MUINT32 variable to be filled with the number of
                                        buffers allowed to be used by the video capture device.

Return Value:   LStatus_OK              Function completed successfully.
                LStatus_INVALID_PARAM   Function failed due to invalid parameters.
                LStatus_FAIL            Function failed.

Comments:       None.

\************************************************************************************************************/
LAPI LStatus LVideoIn_GetBufferAttributes(
                LVideoIn_Handle hVideoIn, 
                MUINT32*        puiWidth,
                MUINT32*        puiHeight,
                LPixelFormat*   pePixelFormat,
                MUINT32*        puiNbBuffers);


/************************************************************************************************************\

Function:       LVideoIn_StartCapture

Description:    Starts the video capture.

Parameters:     IN  hVideoIn            Handle to the video input device.

Return Value:   LStatus_OK                  Function completed successfully.
                LStatus_INVALID_PARAM       Function failed due to invalid parameters.
                LStatus_FAIL                Function failed.
                LStatus_ACCESS_DENIED       The hVideoIn handle is READONLY.  Read/write access
                                            (LAccessMode_READWRITE) is needed to call this function.
                LStatus_OUT_OF_RESOURCES    Function failed due to insufficient descriptors or bandwidth.

Comments:       - LVideoIn_StartCapture may fail if:
                    1. VideoParameters aren't detected, or the source has changed:
                       LVideoIn_GetDetectedVideoParameters(...) returns LStatus_NOT_OPTIMAL.
                    2. No source is detected:
                       LVideoIn_GetDetectedVideoParameters(...) obtains LVideo_VidParam.uiPixelClock_khz=0.
                    3. The buffer dimensions of VideoIn don't fit with the detected VideoParameters:
                       LVideoIn_GetBufferAttributes() obtains dimensions different than
                       LVideoIn_GetDetectedVideoParameters(...) LVideo_VidParam.(uiVActive uiHActive).
                - The number of capture buffers the video input device is allowed to use can be set via
                  LVideoIn_CreateBuffers.

\************************************************************************************************************/
LAPI LStatus LVideoIn_StartCapture(
                LVideoIn_Handle hVideoIn);


/************************************************************************************************************\

Function:       LVideoIn_StopCapture

Description:    Stops the video capture.

Parameters:     IN  hVideoIn            Handle to the video input device.

Return Value:   LStatus_OK              Function completed successfully.
                LStatus_INVALID_PARAM   Function failed due to invalid parameters.
                LStatus_FAIL            Function failed.
                LStatus_ACCESS_DENIED   The hVideoIn handle is READONLY.  Read/write access
                                        (LAccessMode_READWRITE) is needed to call this function.

Comments:       - Captured video buffers are discarded if not completed when the capture is stopped.

\************************************************************************************************************/
LAPI LStatus LVideoIn_StopCapture(
                LVideoIn_Handle hVideoIn);


/************************************************************************************************************\

Function:       LVideoIn_GetNextBuffer

Description:    Gets the next video buffer.

Parameters:     IN  hVideoIn            Handle to the video input device.
                IN  bWaitBufferReady    MTRUE to wait for a Ready video buffer, MFALSE to not wait. 
                IN  uiTimeout_ms        Timeout, in milliseconds, to wait. 0 means no wait and
                                        LINFINITE_TIMEOUT means wait forever.
                OUT phVideoBuffer       Pointer to an LBuffer_Handle variable to be filled with the handle to 
                                        video buffer.
                OUT puiFieldCounter     Pointer to an MUINT64 variable to be filled with the field counter
                                        value when the returned buffer was captured or will be capture by
                                        the video input. Can be MNULL if this counter value isn't requested.
                OUT puiTickRefCounter   Pointer to an MUINT64 variable to be filled with the tick reference
                                        counter value, in ticks since the system was started, when the returned
                                        buffer was captured by the video input. Return 0 when bWaitBufferReady
                                        is MFALSE and there's no Ready buffer. Can be MNULL if this counter
                                        value isn't requested.
                OUT pbIsTopField        Pointer to an MBOOL32 variable to be filled with MTRUE if the returned
                                        capture buffer corresponds to the top field, MFALSE if it corresponds
                                        to the bottom field. Can be MNULL if a captured video buffer order
                                        isn't requested.

Return Value:   LStatus_OK              Function completed successfully.
                LStatus_INVALID_PARAM   Function failed due to invalid parameters.
                LStatus_FAIL            Function failed.
                LStatus_TIMEOUT         Function exited after a timeout.
                LStatus_CANCELED        Function exited. No capture in progress.
                LStatus_ACCESS_DENIED   The hVideoIn handle is READONLY.  Read/write access
                                        (LAccessMode_READWRITE) is needed to call this function.

Comments:       - The video capture buffers returned are allocated and owned by the video input device. 
                  They're loaned to the application for processing. The capture buffer state changes from 
                  Released to Loaned when it's loaned to the application.
                - When bWaitBufferReady is set to MFALSE, the buffer is return even if it's not Ready or
                  completely filled with the video capture data. In this case, uiTickRefCounter will be
                  set to 0.
                - The application can schedule video processing in advance on Empty buffers or on parts of 
                  Empty buffers in low latency mode.  To get the handle of the Empty buffers, call the 
                  LVideoIn_GetNextBuffer service setting bWaitBufferReady to MFALSE.  The call will then
                  return the handle of an Empty capture buffer on which processing can be scheduled in a 
                  device thread. To synchronized the processing and avoid using the Empty capture buffer 
                  before it actually contains captured fields, use the LVideoIn_ScheduleWaitFieldCounter with
                  the returned value return in the puiFieldCounter.
                - A loaned buffer must be released using the LVideoIn_ReleaseBuffer or 
                  LVideoIn_ScheduleReleaseBuffer services after the application processed them to allow the 
                  video input device to re-use the buffers to capture new fields.
                - If the capture buffers aren't released, the video input device will run out of Empty 
                  buffers and hold the last empty buffers available to capture until a buffer is released.
                - The field counter is incremented each time a new capturable field from the input source is 
                  seen by the video input device, whether or not the field is actually captured or dropped.
                - If two consecutive buffers, returned by the LVideoIn_GetNextBuffer service, have a 
                  difference greater than one between their associated field counter values, some input source 
                  fields were dropped (i.e. have not been captured to capture buffers). This can occur if 
                  capture buffers were unavailable at the time. To know how many input source fields haven't 
                  been captured, subtract 1 from the field counter values difference.
                - If interlaced video parameters are used, two (2) fields are required for a whole frame to 
                  be captured while only one (1) field per frame is required with progressive video parameters.
                - If interlaced video parameters are used, pbIsTopField can be used to know if the top or 
                  bottom field has been captured.
                - If non-interlaced video parameters are used, pbIsTopField is always set to MTRUE as there's 
                  a single field and no top or bottom fields.

\************************************************************************************************************/
LAPI LStatus LVideoIn_GetNextBuffer(
                LVideoIn_Handle hVideoIn,
                MBOOL32         bWaitBufferReady,
                MUINT32         uiTimeout_ms,
                LBuffer_Handle* phVideoBuffer,
                MUINT64*        puiFieldCounter,
                MUINT64*        puiTickRefCounter,
                MBOOL32*        pbIsTopField);


/************************************************************************************************************\

Function:       LVideoIn_ReleaseBuffer

Description:    Releases a video capture buffer.

Parameters:     IN  hVideoIn            Handle to the video input device.
                IN  hVideoBuffer        Handle to the video buffer to release.

Return Value:   LStatus_OK              Function completed successfully.
                LStatus_INVALID_PARAM   Function failed due to invalid parameters.
                LStatus_FAIL            Function failed.
                LStatus_ACCESS_DENIED   The hVideoIn handle is READONLY.  Read/write access
                                        (LAccessMode_READWRITE) is needed to call this function.

Comments:       - Capture video buffers loaned via the LVideoIn_GetNextBuffer service must be released after 
                  the application processes them to allow the video input device to re-use it to capture other 
                  input source fields. The number of capture buffers for a given video input device is limited. 
                  Not releasing video capture buffers starves the video input device and will eventually force 
                  it to temporarily stop capturing input source fields.
                - The LVideoIn_ReleaseBuffer service changes the states of a capture buffer to 'Empty' and 
                  'Released'.
                - The buffers can be released out of order over GetNextBuffer.

\************************************************************************************************************/
LAPI LStatus LVideoIn_ReleaseBuffer(
                LVideoIn_Handle     hVideoIn,
                LBuffer_Handle      hVideoBuffer);


/************************************************************************************************************\

Function:       LVideoIn_ScheduleReleaseBuffer

Description:    Schedules the release of a video capture buffer of the video input device by adding required 
                commands to the passed in device thread object.

Parameters:     IN  hVideoIn            Handle to the video input device.
                IN  hVideoBuffer        Handle to the video buffer to release.
                IN  hDeviceThread       Handle to the device thread to use to schedule the operation.
                IN  uiTimeout_ms        Device thread full queue wait timeout.

Return Value:   LStatus_OK              Function completed successfully.
                LStatus_INVALID_PARAM   Function failed due to invalid parameters.
                LStatus_FAIL            Function failed.
                LStatus_ACCESS_DENIED   The hVideoIn handle is READONLY.  Read/write access
                                        (LAccessMode_READWRITE) is needed to call this function.
                LStatus_TIMEOUT         Function exited after a timeout.

Comments:       - Capture video buffers loaned via the LVideoIn_GetNextBuffer service must be released after 
                  the application processes them to allow the video input device to re-use it to capture other 
                  input source fields. The number of capture buffers for a given video input device is limited. 
                  Not releasing video capture buffers starves the video input device and will eventually force 
                  it to temporarily stop capturing input source fields.
                - The LVideoIn_ScheduleReleaseBuffer service changes the state of a capture buffer from 
                  Loaned to Empty.
                - LStatus_TIMEOUT is returned when the ReleaseBuffer command can't be added to the device 
                  thread because it's full and no room becomes available before the timeout period expires.
                - The buffers can be released out of order using GetNextBuffer.


\************************************************************************************************************/
LAPI LStatus LVideoIn_ScheduleReleaseBuffer(
                LVideoIn_Handle         hVideoIn,
                LBuffer_Handle          hVideoBuffer,
                LDeviceThread_Handle    hDeviceThread,
                MUINT32                 uiTimeout_ms);


/************************************************************************************************************\

Function:       LVideoIn_ScheduleWaitFieldCounter

Description:    Schedules a wait of a field counter of the video input device by adding required commands to 
                the passed in device thread object.

Parameters:     IN  hVideoIn            Handle to the video input device.
                IN  hDeviceThread       Handle to the device thread to use to schedule the operation.
                IN  uiTimeout_ms        Device thread full queue wait timeout.
                IN  uiFieldCounter      Field counter value when the buffer is captured by 
                                        the video input.

Return Value:   LStatus_OK              Function completed successfully.
                LStatus_INVALID_PARAM   Function failed due to invalid parameters.
                LStatus_FAIL            Function failed.
                LStatus_TIMEOUT         Function exited after a timeout.


Comments:       - LStatus_TIMEOUT is returned when the wait command can't be added to the device thread 
                  because it's full and no room becomes available before the timeout period expires.

\************************************************************************************************************/
LAPI LStatus LVideoIn_ScheduleWaitFieldCounter(
                LVideoIn_Handle         hVideoIn,
                LDeviceThread_Handle    hDeviceThread,
                MUINT32                 uiTimeout_ms,
                MUINT64                 uiFieldCounter);


/************************************************************************************************************\

Function:       LVideoIn_DiscardReadyBuffers

Description:    Discards all capture buffers in the Ready state.

Parameters:     IN  hVideoIn            Handle to the video input device.

Return Value:   LStatus_OK              Function completed successfully.
                LStatus_INVALID_PARAM   Function failed due to invalid parameters.
                LStatus_FAIL            Function failed.
                LStatus_ACCESS_DENIED   The hVideoIn handle is READONLY.  Read/write access
                                        (LAccessMode_READWRITE) is needed to call this function.

Comments:       - Capture buffers in Ready state contain captured data not yet retrieved by the 
                  application.
                - The state of all discarded capture buffers is set to Empty and marks them as available to 
                  be used to capture new input source fields.
                - The LVideoIn_DiscardReadyBuffers service can be used to discard any unprocessed captured 
                  buffers before re-starting the capture.
                - All capture buffers loaned to the application or currently in the Empty state are 
                  unaffected by this service.

\************************************************************************************************************/
LAPI LStatus LVideoIn_DiscardReadyBuffers(
                LVideoIn_Handle hVideoIn);


/************************************************************************************************************\

Function:       LVideoIn_GetCaptureState

Description:    Return the video capture state.

Parameters:     IN  hVideoIn                Handle to the video input device.
                OUT peState                 Pointer to an LVideoIn_State variable to be filled with 
                                             the current capture state.
                OUT puiTotalBufferCount     Pointer to an MUINT32 variable to be filled with the currently
                                             allocated capture buffers count.  
                OUT puiReadyBufferCount     Pointer to an MUINT32 variable to be filled with the currently
                                             ready capture buffers count.
                OUT puiLoanedBufferCount    Pointer to an MUINT32 variable to be filled with the currently
                                             loaned capture buffers count.
                OUT puiInternalBufferCount  Pointer to an MUINT32 variable to be filled with the number of
                                             internally created buffers to make the hardware run.
                                             These buffers aren't part of the total buffer count.

Return Value:   LStatus_OK                  Function completed successfully.
                LStatus_INVALID_PARAM       Function failed due to invalid parameters.
                LStatus_FAIL                Function failed.

Comments:       - For more information on buffer counts see LVideoIn_GetNextBuffer.

\************************************************************************************************************/
LAPI LStatus LVideoIn_GetCaptureState(
                LVideoIn_Handle hVideoIn,
                LVideoIn_State* peState,
                MUINT32*        puiTotalBufferCount,
                MUINT32*        puiReadyBufferCount,
                MUINT32*        puiLoanedBufferCount,
                MUINT32*        puiInternalBufferCount);

/************************************************************************************************************\

Function:       LVideoIn_WaitForFieldStart

Description:    Blocks a CPU thread until the next field capture starts.

Parameters:     IN  hVideoIn            Handle to the video input device.
                IN  uiTimeout_ms        Time out to wait in milliseconds.  0 means no wait and 
                                        LINFINITE_TIMEOUT means forever.
                OUT puiFieldCounter     Pointer to an MUINT64 variable to be filled with the field number of 
                                        the just started field. Can be MNULL if the field counter isn't 
                                        requested.
                OUT puiTickRefCounter   Pointer to an MUINT64 variable to be filled with the tick reference 
                                        counter value, in ticks since the system was started, when the 
                                        field counter was updated.  Can be MNULL if this counter value is 
                                        not requested.
                OUT pbIsTopField        Pointer to an MBOOL32 variable to be filled with MTRUE if the top 
                                        field is being captured, MFALSE if it is the bottom field.  Can 
                                        be MNULL if the captured field isn't requested.

Return Value:   LStatus_OK              Function completed successfully.
                LStatus_INVALID_PARAM   Function failed due to invalid parameters.
                LStatus_FAIL            Function failed.
                LStatus_TIMEOUT         Function exited after a timeout.
                LStatus_CANCELED        Function exited. No capture in progress.
                LStatus_UNSUPPORTED     Function aborted because waiting for field start is unsupported 
                                         by the current hardware configuration.

Comments:       - See LVideoIn_GetNextBuffer service for more details about field and tick counters or
                  about top field.

\************************************************************************************************************/
LAPI LStatus LVideoIn_WaitForFieldStart(
                LVideoIn_Handle hVideoIn,
                MUINT32         uiTimeout_ms,
                MUINT64*        puiFieldCounter,
                MUINT64*        puiTickRefCounter,
                MBOOL32*        pbIsTopField);


/************************************************************************************************************\

Function:       LVideoIn_GetLastFieldCounter

Description:    Gets the last captured field counters value. 

Parameters:     IN  hVideoIn            Handle to the video input device.
                OUT puiFieldCounter     Pointer to an MUINT64 variable to be filled with the last field counter
                                        number.  Can be MNULL if the captured field isn't requested.
                OUT puiTickRefCounter   Pointer to an MUINT64 variable to be filled with the tick reference 
                                        counter value, in ticks since the system was started, when the 
                                        field counter was last updated.  Can be MNULL if this counter value 
                                        isn't requested.
                OUT pbIsTopField        Pointer to an MBOOL32 variable to be filled with MTRUE if the top 
                                        field is being captured, MFALSE if it is the bottom field.  Can 
                                        be MNULL if displayed field isn't requested.

Return Value:   LStatus_OK              Function completed successfully.
                LStatus_INVALID_PARAM   Function failed due to invalid parameters.
                LStatus_FAIL            Function failed.

Comments:       - For more information on field counters, tick counters, and top fields, 
                  see LVideoIn_GetNextBuffer.

\************************************************************************************************************/
LAPI LStatus LVideoIn_GetLastFieldCounter(
                LVideoIn_Handle hVideoIn,
                MUINT64*        puiFieldCounter,
                MUINT64*        puiTickRefCounter,
                MBOOL32*        pbIsTopField);

/************************************************************************************************************\

Function:       LVideoIn_GetFieldCounterLUID

Description:    Gets captured field counters unique identifiers (LUID). 

Parameters:     IN  hVideoIn            Handle to the video input device.
                OUT puiFieldCounterLUID Pointer to an MUINT64 variable where the fields counter unique 
                                         identifier is written.

Return Value:   LStatus_OK              Function completed successfully.
                LStatus_INVALID_PARAM   Function failed due to invalid parameters.
                LStatus_FAIL            Function failed.

Comments:       - Use to synchronizes the device thread with the completion of a counter.

\************************************************************************************************************/
LAPI LStatus LVideoIn_GetFieldCounterLUID(
                LVideoIn_Handle     hVideoIn,
                MUINT64*            puiFieldCounterLUID);


/************************************************************************************************************\

Function:       LVideoIn_WaitForEvent

Description:    Blocks a CPU thread until VideoIn events occur. This function does not miss any events 
                that occurred since the VideoIn handle creation.

Parameters:     IN  hVideoIn                Handle to the video input device.
                IN  uiTimeout_ms            Time out to wait in milliseconds.  0 means no wait and 
                                            LINFINITE_TIMEOUT means forever.
                OUT peEventsSignaled        Pointer to an LVideoIn_Event variable to be filled with
                                            capture events. Only valid when LStatus_OK is returned.

Return Value:   LStatus_OK                  Function completed successfully.
                LStatus_INVALID_PARAM       Function failed due to invalid parameters.
                LStatus_FAIL                Function failed.
                LStatus_UNSUPPORTED         Waiting for capture events isn't supported.
                LStatus_TIMEOUT             Function exited after a timeout.

Comments:       - A capture event can be used to notify of important events during a capture session.
                - Wait for capture events can be set before the capture is started.
                - We recommend creating a handle for each thread that uses this function.
                - This function doesn't miss any events that occurred since the hVideoIn handle creation.
                  If two events are signaled before the called of LVideoIn_WaitForEvent and after the
                  LVideoIn_GetHandle, LVideoIn_WaitForEvent will return the immediately two times with
                  the corresponding event code.

\************************************************************************************************************/
LAPI LStatus LVideoIn_WaitForEvent(
                LVideoIn_Handle hVideoIn,
                MUINT32         uiTimeout_ms,
                LVideoIn_Event* peEventsSignaled);


/************************************************************************************************************\

Function:       LVideoIn_GetControlInfo

Description:    Returns information about a video input device control.

Parameters:     IN  hVideoIn            Handle to the video input device.
                IN  eControlId          ID of the control for which information should be returned.  
                                        Must be a value from the LVideoIn_ControlId.
                OUT poControlInfo       Pointer to an LVideoIn_ControlInfo object to be filled with the
                                        specified video input device control information.

Return Value:   LStatus_OK              Function completed successfully.
                LStatus_INVALID_PARAM   Function failed due to invalid parameters.
                LStatus_FAIL            Function failed.
                LStatus_UNSUPPORTED     ID of a control that's unsupported by the current hardware
                                         configuration was specified.

Comments:       - For more information on video input device controls, see LVideoIn_ControlId and 
                  LVideoIn_ControlInfo.

\************************************************************************************************************/
LAPI LStatus LVideoIn_GetControlInfo(
                LVideoIn_Handle         hVideoIn,
                LVideoIn_ControlId      eControlId,
                LVideoIn_ControlInfo*   poControlInfo);


/************************************************************************************************************\

Function:       LVideoIn_GetControlValues

Description:    Gets the value of the specified video input device controls.

Parameters:     IN  hVideoIn            Handle to the video input device.
                IN  uiCount             Number of video input device controls value to get.
                IN  paeControlIds       Pointer to an array of LVideoIn_ControlId variables containing the 
                                        IDs of all the controls for which their current value should be 
                                        returned.  The array must have at least 'uiCount' entries to avoid 
                                        memory overruns.
                OUT paiControlValues    Pointer to an array of MINT32 variables to be filled with the 
                                        specified controls current value.  The array must have at least 
                                        'uiCount' entries to avoid memory overruns.

Return Value:   LStatus_OK              Function completed successfully.
                LStatus_INVALID_PARAM   Function failed due to invalid parameters.
                LStatus_FAIL            Function failed.
                LStatus_UNSUPPORTED     At least one control ID specified is unsupported by the current
                                        hardware configuration.

Comments:       - For more information on video input device controls, see LVideoIn_ControlId and 
                  LVideoIn_ControlInfo.

\************************************************************************************************************/
LAPI LStatus LVideoIn_GetControlValues(
                LVideoIn_Handle     hVideoIn,
                MUINT32             uiCount,
                LVideoIn_ControlId* paeControlIds,
                MINT32*             paiControlValues);


/************************************************************************************************************\

Function:       LVideoIn_SetControlValues

Description:    Sets the value of the specified video input device controls.

Parameters:     IN  hVideoIn            Handle to the video input device.
                IN  uiCount             Number of video input device controls values to set.
                IN  paeControlIds       Pointer to an array of LVideoIn_ControlId variables containing the 
                                        Ids of all the controls for which their value should be set.  The 
                                        array must have at least 'uiCount' entries to avoid memory overruns.
                OUT paiControlValues    Pointer to an array of MINT32 variables containing the new controls 
                                        value to set.  The array must have at least 'uiCount' count entries 
                                        to avoid memory overruns.

Return Value:   LStatus_OK              Function completed successfully.
                LStatus_INVALID_PARAM   Function failed due to invalid parameters.
                LStatus_FAIL            Function failed.
                LStatus_ACCESS_DENIED   The hVideoIn handle is READONLY.  Read/write access
                                        (LAccessMode_READWRITE) is needed to call this function.
                LStatus_UNSUPPORTED     At least one control Id specified isn't supported by the current 
                                        hardware configuration.  No control value have been set.

Comments:       - For more information on video input device controls, see LVideoIn_ControlId and 
                  LVideoIn_ControlInfo.
                - LStatus_INVALID_PARAM is returned if any of the passed in control values are invalid and 
                  the control value remains unchanged. An invalid control value is a value smaller than the 
                  minimal value, greater than the maximum value, or not a multiple of the step value.

\************************************************************************************************************/
LAPI LStatus LVideoIn_SetControlValues(
                LVideoIn_Handle     hVideoIn,
                MUINT32             uiCount,
                LVideoIn_ControlId* paeControlIds,
                MINT32*             paiControlValues);

/************************************************************************************************************\

Function:       LVideoIn_GetAviInfoFrameHeader

Description:    Gets the AVI info frame header.

Parameters:     IN      hVideoIn                Handle to the video input device.
                OUT     poAviInfoFrameHeader    Pointer to a LVideoIn_InfoFrameHeader structure to be filled
                                                with the header data of the AVI InfoFrame (see 
                                                LVideoIn_InfoFrameHeader for detail).

Return Value:   LStatus_OK              Function completed successfully.
                LStatus_INVALID_PARAM   Function failed due to invalid parameters.
                LStatus_FAIL            Function failed.
                LStatus_UNSUPPORTED     AVI InfoFrame feature isn't supported with the current video input.
                LStatus_NOT_OPTIMAL     Input source has changed since last DetectSource has been called,
                                        video parameters does not represent the current state.

Comments:       None.

\************************************************************************************************************/
LAPI LStatus LVideoIn_GetAviInfoFrameHeader(
                LVideoIn_Handle             hVideoIn,
                LVideoIn_InfoFrameHeader*   poAviInfoFrameHeader);

/************************************************************************************************************\

Function:       LVideoIn_GetAviInfoFrameRawData

Description:    Gets the AVI InfoFrame raw data, excluding the InfoFrame header.

Parameters:     IN      hVideoIn            Handle to the video input device.
                IN      uiSize              Size, in bytes, of the AVI InfoFrame raw data buffer.
                OUT     pauiInfoFrameData   Pointer to an AVI InfoFrame raw data buffer where the AVi 
                                            InfoFrame raw data is written (this buffer should be allocated
                                            before passing to this function).

Return Value:   LStatus_OK              Function completed successfully.
                LStatus_INVALID_PARAM   Function failed due to invalid parameters.
                LStatus_FAIL            Function failed.
                LStatus_UNSUPPORTED     AVI InfoFrame feature isn't supported with the current video input.
                LStatus_NOT_OPTIMAL     Input source has changed since last DetectSource has been called,
                                        video parameters does not represent the current state.

Comments:       None.

\************************************************************************************************************/
LAPI LStatus LVideoIn_GetAviInfoFrameRawData(
                LVideoIn_Handle     hVideoIn,
                MUINT32             uiSize,
                MUINT8*             pauiInfoFrameData);

/************************************************************************************************************\

Function:       LVideoIn_GetDescription

Description:    Gets the video input description.

Parameters:     IN  hDev                Device handle.
                IN  uiIndex             Index of the independent video input for which the description 
                                        should be returned.
                OUT szDescription       Buffer where the ANSI null terminated video input description is
                                        written.

Return Value:   LStatus_OK              Function completed successfully.
                LStatus_INVALID_PARAM   Function failed due to invalid parameters.
                LStatus_FAIL            Function failed.

Comments:       None.

\************************************************************************************************************/
LAPI LStatus LVideoIn_GetDescription(
                LDevice_Handle      hDev,
                MUINT32             uiIndex,
                LSTR64              szDescription);

/************************************************************************************************************\

Function:       LVideoIn_SetVideoParametersOverride

Description:    Sets the video parameters that are going to be used for the capture. Each time
                LVideoIn_DetectSource is successfully executed, these video parameters set to
                the detected video parameters.

Parameters:     IN  hVideoIn            Handle to the video input device.
                IN  poVidParam          Pointer to a LVideo_VidParam variable filled with the desired video
                                        parameters.

Return Value:   LStatus_OK              Function completed successfully.
                LStatus_INVALID_PARAM   Function failed due to invalid parameters.
                LStatus_FAIL            Function failed.

Comments:       - May be used is situations where the video signal isn't properly detected. In this case,
                  LVideoIn_DetectSource should be called anyway, since it allows the hardware to be correctly
                  configured  for the capture.

\************************************************************************************************************/
LAPI LStatus LVideoIn_SetVideoParametersOverride(
                LVideoIn_Handle     hVideoIn,
                LVideo_VidParam*    poVidParam);

/************************************************************************************************************\

Function:       LVideoIn_GetVideoParametersOverride

Description:    Gets the video parameters that are going to be used for the capture. These parameters may be
                changed using LVideoIn_SetVideoParametersOverride. Each time LVideoIn_DetectSource is
                successfully executed, these video parameters are overwritten with the detected video
                parameters.

Parameters:     IN  hVideoIn            Handle to the video input device.
                OUT poVidParam          Pointer to a LVideo_VidParam variable to be filled with the
                                        overwritten video parameters.

Return Value:   LStatus_OK              Function completed successfully.
                LStatus_INVALID_PARAM   Function failed due to invalid parameters.
                LStatus_FAIL            Function failed.

Comments:       - None.

\************************************************************************************************************/
LAPI LStatus LVideoIn_GetVideoParametersOverride(
                LVideoIn_Handle     hVideoIn,
                LVideo_VidParam*    poVidParam);

/************************************************************************************************************\

Function:       LVideoIn_SetCreateBuffersOptions

Description:    Sets video buffer options. The buffer options are used in LVideoIn_CreateBuffers until
                LVideoIn_Handle is released.

Parameters:     IN  hVideoIn            Handle to the video input device.
                IN  poOptions           Pointer to an LBuffer_Options object containing options to be
                                        use in LVideoIn_CreateBuffers.

Return Value:   LStatus_OK              Function completed successfully.
                LStatus_INVALID_PARAM   Function failed due to invalid parameters.
                LStatus_FAIL            Function failed.

Comments:       - Optional function. No options are set by default.
                - Used to enable the auto-sync feature.

\************************************************************************************************************/
LAPI LStatus LVideoIn_SetCreateBuffersOptions(
                LVideoIn_Handle     hVideoIn, 
                LBuffer_Options*    poOptions);

#ifdef EXTERNAL_HDCP_SUPPORT
/************************************************************************************************************\

Function:       LVideoIn_SetEncryptionContentKey

Description:    Sets the encryption content key.

Parameters:     IN  hVideoIn            Handle to the video input device.
                IN  poContentKey        Pointer to an LSecureEngine_ContentKey object containing the key for
                                        encryption.

Return Value:   LStatus_OK              Function completed successfully.
                LStatus_INVALID_PARAM   Function failed due to invalid parameters.
                LStatus_FAIL            Function failed.

Comments:       None.

\************************************************************************************************************/
LAPI LStatus LVideoIn_SetEncryptionContentKey(
                    LVideoIn_Handle             hVideoIn,
                    LSecureEngine_ContentKey*   poContentKey);
#endif //EXTERNAL_HDCP_SUPPORT

/************************************************************************************************************\

Function:       LVideoIn_ValidateVideoParameters

Description:    Tests the videos parameters for a video input.

Parameters:     IN  hVideoIn            Handle to the video input device.
                IN  eSignalType         Video signal type for which to test the video parameters.
                IN  poVidParam          Pointer to an LVideo_VidParam object containing the video parameters
                                        to validate.

Return Value:   LStatus_OK              Function completed successfully.
                LStatus_INVALID_PARAM   Function failed due to invalid parameters.
                LStatus_UNSUPPORTED     Function failed because the hardware doesn't support the
                                        video parameters or the pixel format for the video parameters.
                LStatus_FAIL            Function failed.

Comments:       None.

\************************************************************************************************************/
LAPI LStatus LVideoIn_ValidateVideoParameters(
                    LVideoIn_Handle             hVideoIn,
                    LVideo_SignalType           eSignalType,
                    LVideo_VidParam*            poVidParam);

#if defined (__cplusplus)
}
#endif

#endif // #ifndef INC_LVIDEOIN_H
